{===============================================================================
ChatInterface.pas
	Battle.net Chat interpreter component
	written by Sarah Davis and Dan Stipp.
	send questions or comments to topazchat@hotmail.com

Use this component together with a TChatInterface component to manage a
Battle.net Chat Session.  First drop one ChatInterface component on the form
(this component was derived from a Windowed Control rather than a TComponent
to allow it to receive Winsock Messages).  Then drop a TBnetChat component
on the form, and set it's ChatInterface property to the ChatInterface you
placed on the form.  Use the Connect command of the ChatInterface to connect
and the Send Method of the BNetChat component to Send data (you may send
data directly through the ChatInterface, but the Send method of the BnetChat
component handles anti-flood delays automatically).  Be sure to send a Return
Character (#13) when a line is finished or Battle.net will not process the
text.

When Data is recieved, the OnDataReceived event of the ChatInterface is called.
In this method, you call the ProcessString method of any Interpreter (TBNetChat,
TTextChat, or any other interpreter you derive from TBaseInterpreter), and
pass the DataString to the Interpreter.  Example:

procedure TfrmChat.ChatInterfaceDataReceived(DataString: String);
begin
	BNetChat.ProcessString(DataString);
end;

The BnetChat does NOT do any communication with Winsock, it merely interprets
Data from the ChatInterface.  The ProcessString method breaks down Battle.net
commands and calls the events based on data received by the chat interface.

Once connected, use the Login method of the TBnetChat component rather than
the ChatInterface version, as it sends the necessary codes to Battle.net to
establisha a connection, including the name and password defined in the UserName
and Password properties.
===============================================================================}
unit Interpreters;

interface

uses
	Classes, SysUtils, ChatInterface, ExtCtrls, StdCtrls, Messages, Windows;

Const

{===============================================================================
Battle.net specific constants
===============================================================================}
	bnShowuser           =1001;
	bnJoin               =1002;
	bnLeave              =1003;
	bnWhisper            =1004;
	bnTalk               =1005;
	bnBroadcast          =1006;
	bnChannel            =1007;
	bnUserFlags          =1009;
	bnWhisperSent        =1010;
	bnChannelFull        =1013;
	bnChannelNotExist		=1014;
	bnChannelRestricted  =1015;
	bnInfo               =1018;
	bnError              =1019;
	bnEmote              =1023;
	bnUniqueName         =2010;
	bnNull				   =2000;
	bnUnknown				=0;
	bnDiabloRetail			='DRTL';
	bnStarRetail 			='STAR';
	bnDiabloShare			='DSHR';
	bnStarShare				='SSHR';
	bnChat		 			='CHAT';
	bnStarBeta				='SEXP';

type
	tBaseEvent=			procedure(Mess: String) of Object;
	tUserEvent=			procedure(User: String) of Object;
	tTalkEvent=			procedure(User,Mess: String) of object;
	tUserFlagEvent=	procedure(User,Flags: String) of Object;
	tUserTypeEvent=	procedure(User,Flags,UserType: String) of Object;
	tBaseInterpreter=Class(tComponent)
	private
		fOnTalk:				tTalkEvent;
		fOnWhisper:			tTalkEvent;
		fOnWhisperSent:	tTalkEvent;
		fOnShowUser:		tUserTypeEvent;
		fOnInfo:				tBaseEvent;
		fOnJoin:				tUserTypeEvent;
		fOnLeave:			tUserTypeEvent;
		fOnBroadcast:		tBaseEvent;
		fOnChannel:			tBaseEvent;
		fOnUserFlags:		tUserFlagEvent;
		fOnChannelFull:	tBaseEvent;
		fOnChanNotExist: 	tBaseEvent;
		fOnRestricted:		tBaseEvent;
		fOnError:			tBaseEvent;
		fOnEmote:			tTalkEvent;
		fOnUniqueName:		tUserEvent;
		fOnUnHandled:		tBaseEvent;
		fOnData:				tBaseEvent;
		fOnNull:				tBaseEvent;
		fOnUnknown:			tBaseEvent;
		fChatInterface:	tChatInterface;
		function		GetUser(St: String): String; Virtual; Abstract;
		function		GetMessage(St: String): String; Virtual; Abstract;
		function		GetFlags(St: String): String; Virtual; Abstract;
		function		GetUserType(St: String): String; Virtual; Abstract;
	public
		EchoOnSend:			Boolean;
		Constructor Create(aOwner: tComponent); Override;
		Destructor  Destroy; Override;
		procedure 	ProcessString(St: String); Virtual; Abstract;
		procedure	Send(St: String); Virtual; Abstract;
	protected
		property OnTalk: tTalkEvent
			read  fOnTalk
			write fOnTalk;
		property OnWhisper: tTalkEvent
			read	fOnWhisper
			write fOnWhisper;
		property OnWhisperSent: tTalkEvent
			read  fOnWhisperSent
			write fOnWhisperSent;
		property OnShowUser:	tUserTypeEvent
			read  fOnShowUser
			write fOnShowUser;
		property OnInfo: tBaseEvent
			read  fOnInfo
			write fOnInfo;
		property OnJoin: tUserTypeEvent
			read  fOnJoin
			write fOnJoin;
		property OnLeave: tUserTypeEvent
			read  fOnLeave
			write fOnLeave;
		property OnBroadcast: tBaseEvent
			read  fOnBroadcast
			write fOnBroadcast;
		property OnChannel: tBaseEvent
			read  fOnChannel
			write fOnChannel;
		property OnUserFlags: tUserFlagEvent
			read  fOnUserFlags
			write fOnUserFlags;
		property OnChannelFull:	tBaseEvent
			read  fOnChannelFull
			write fOnChannelFull;
		property OnChanNotExist: tBaseEvent
			read  fOnChanNotExist
			write fOnChanNotExist;
		property OnRestricted: tBaseEvent
			read  fOnRestricted
			write fOnRestricted;
		property OnError: tBaseEvent
			read  fOnError
			write fOnError;
		property OnEmote: tTalkEvent
			read  fOnEmote
			write fOnEmote;
		property OnUniqueName: tUserEvent
			read  fOnUniqueName
			write fOnUniqueName;
		property OnUnHandled: tBaseEvent
			read  fOnUnHandled
			write fOnUnHandled;
		property OnData: tBaseEvent
			read  fOnData
			write fOnData;
		property OnNull: tBaseEvent
			read  fOnNull
			write	fOnNull;
		property OnUnknown: tBaseEvent
			read  fOnUnknown
			write fOnUnknown;
	published
		property	ChatInterface: TChatInterface
			read	fChatInterface
			write	fChatInterface;
	end;
	TBnetChat=Class(tBaseInterpreter)
	private
		fTimer:				tTimer;
		fMsgQ:				tStringList;
		fOKToSend:			Boolean;
		fUserName:			String;
		fPassword:			String;
		procedure	fDoTimer(Sender: tObject);
		function		GetUser(St: String): String; Override;
		function		GetMessage(St: String): String; Override;
		function		GetFlags(St: String): String; Override;
		function		GetUserType(St: String): String; Override;
		function		GetInterval: Integer;
		procedure 	SetInterval(Value: integer);
	public
		Constructor Create(aOwner: tComponent); override;
		Destructor  Destroy; override;
		procedure 	ProcessString(St: String); override;
		procedure	Send(St: String); Override;
		function		LogIn: Boolean;
	published
		property OnTalk;
		property OnWhisper;
		property OnWhisperSent;
		property OnShowUser;
		property OnInfo;
		property OnJoin;
		property OnLeave;
		property OnBroadcast;
		property OnChannel;
		property OnUserFlags;
		property OnChannelFull;
		property OnChanNotExist;
		property OnRestricted;
		property OnError;
		property OnEmote;
		property OnUniqueName;
		property OnUnHandled;
		property FloodInterval: integer
			read  GetInterval
			write SetInterval;
		property Password: String
			read	fPassword
			write fPassword;
		property UserName: String
			read 	fUserName
			write fUserName;
	end;
	TTextChat=Class(tBaseInterpreter)
	private
	public
		Constructor Create(aOwner: tComponent); override;
		procedure 	ProcessString(St: String); override;
		procedure 	Send(St: String); override;
	published
		property OnData;
	end;
	TTabEdit=Class(tComboBox)
	private
		procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
	end;

procedure Register;

implementation

{===============================================================================
Base interpreter methods
===============================================================================}
Constructor tBaseInterpreter.Create(aOwner: tComponent);
begin
	Inherited Create(aOwner);
	fOnTalk:=nil;
	fOnWhisper:=nil;
	fOnWhisperSent:=nil;
	fOnShowUser:=nil;
	fOnInfo:=nil;
	fOnJoin:=nil;
	fOnLeave:=nil;
	fOnBroadcast:=nil;
	fOnChannel:=nil;
	fOnUserFlags:=nil;
	fOnChannelFull:=nil;
	fOnChanNotExist:=nil;
	fOnRestricted:=nil;
	fOnError:=nil;
	fOnEmote:=nil;
	fOnUniqueName:=nil;
	fOnUnHandled:=nil;
	fOnData:=Nil;
	fChatInterface:=Nil;
	fOnNull:=Nil;
	fOnUnknown:=Nil;
	EchoOnSend:=True;
end;

Destructor  tBaseInterpreter.Destroy;
begin
	Inherited Destroy;
end;

{===============================================================================
BattleNet interface
===============================================================================}
Constructor tBnetChat.Create(aOwner: tComponent);
begin
	inherited Create(aOwner);
	fTimer:=tTimer.Create(Self);
	fTimer.OnTimer:=fDoTimer;
	fTimer.Interval:=1500;
	fTimer.Enabled:=True;
	fMsgQ:=tStringList.Create;
	fUserName:='Guest';
	fPassword:='';
	fOkToSend:=true;
	EchoOnSend:=False;
end;

Destructor  tBnetChat.Destroy;
begin
	fTimer.Free;
	fMsgQ.Free;
	inherited Destroy;
end;

function tBnetChat.GetInterval: Integer;
begin
	if Assigned(fTimer) then Result:=fTimer.Interval
	else Result:=0;
end;

procedure tBnetChat.SetInterval(Value: integer);
begin
	if Assigned(fTimer) then begin
		fTimer.Interval:=Value;
	end;
end;

procedure	tBnetChat.fDoTimer(Sender: tObject);
begin
	if Assigned(fMsgQ) and Assigned(fChatInterface) then begin
		if not fChatInterface.Connected then fMsgQ.Clear;
		if fMsgQ.Count>0 then begin
			fChatInterface.Send(fMsgQ[0]);
			fMsgQ.Delete(0);
		end else begin
			fOkToSend:=True;
		end;
	end;
end;

function tBnetChat.LogIn: Boolean;
begin
	Result:=False;
	if Assigned(fChatInterface) then begin
		with fChatInterface do begin
			LoginScript.Clear;
			LoginScript.Add(#3#4+fUserName+#13+fPassword+#13);
			Result:=fChatInterface.LogIn;
		end;
	end;
end;

function tBnetChat.GetUser(St: String): String;
var
	UStart: Boolean;
	i: integer;
begin
	Result:='';
	St:=copy(St,6,length(St));
	UStart:=false;
	for i:=1 to length(St) do begin
		if (St[i]=#32) then begin
			if not UStart then begin
				UStart:=true;
			end else begin
				break;
			end;
		end;
		if UStart and (St[i]<>#32) then Result:=Result+St[i];
	end;
end;

function tBnetChat.GetMessage(St: String): String;
var
	i: integer;
begin
	Result:='';
	for i:=1 to length(St) do begin
		if St[i]='"' then begin
			Result:=Copy(St,i+1,Length(St)-i-1);
			break;
		end;
	end;
end;

function tBnetChat.GetFlags(St: String): String;
begin
	Result:='';
	if St[length(St)]=']' then begin
		Result:=copy(St,length(St)-10,4);
	end else begin
		Result:=Copy(St,length(St)-3,4);
	end;
end;

function tBnetChat.GetUserType(St: String): String;
begin
	Result:=Copy(St,length(St)-4,4);
end;

procedure 	tBnetChat.ProcessString(St: String);
var
	Command,Code: integer;
	User,Mess,Flags,UserType:		String;
begin
	Val(Copy(St,1,4),Command,Code);
	if Code<>0 then Command:=bnUnknown;
	User:=GetUser(St);
	Mess:=GetMessage(St);
	Flags:=GetFlags(St);
	UserType:=GetUserType(St);
	Case Command of
		bnShowuser: begin
			if Assigned(fOnShowUser) then fOnShowUser(User,Flags,UserType);
		end;
		bnJoin: begin
			if Assigned(fOnJoin) then fOnJoin(User,Flags,UserType);
		end;
		bnTalk: begin
			if Assigned(fOnTalk) then fOnTalk(User,Mess);
		end;
		bnLeave: begin
			if Assigned(fOnLeave) then fOnLeave(User,Flags,UserType);
		end;
		bnWhisper: begin
			if Assigned(fOnWhisper) then fOnWhisper(User,Mess);
		end;
		bnBroadcast: begin
			if Assigned(fOnbroadcast) then fOnBroadcast(Mess);
		end;
		bnChannel: begin
			if Assigned(fOnChannel) then fOnChannel(Mess);
		end;
		bnUserFlags: begin
			if Assigned(fOnUserflags) then fOnUserFlags(User,Flags);
		end;
		bnWhisperSent: begin
			if Assigned(fOnWhisperSent) then fOnWhisperSent(User,Mess);
		end;
		bnChannelFull: begin
			if Assigned(fOnChannelFull) then fOnChannelFull(Mess);
		end;
		bnChannelNotExist: begin
			if Assigned(fOnChanNotExist) then fOnChanNotExist(Mess);
		end;
		bnChannelRestricted: begin
			if Assigned(fOnRestricted) then fOnRestricted(Mess);
		end;
		bnInfo: begin
			if Assigned(fOnInfo) then fOnInfo(Mess);
		end;
		bnError: begin
			if Assigned(fOnError) then fOnError(Mess);
		end;
		bnEmote: begin
			if Assigned(fOnEmote) then fOnEmote(User,Mess);
		end;
		bnUniqueName: begin
			if Assigned(fOnUniqueName) then fOnUniqueName(User);
		end;
		bnNull: begin
			if Assigned(fOnNull) then fOnNull(Mess);
		end;
		bnUnknown: begin
			if Assigned(fOnUnhandled) then fOnUnhandled(ST);
		end else begin
			if Assigned(fOnUnhandled) then fOnUnhandled(ST);
		end;
	end;
end;

procedure tBnetChat.Send(St: String);
begin
	if Assigned(fChatInterface) then begin
		if fOkToSend then begin
			if fMsgQ.Count<=0 then begin
				fChatInterface.Send(St);
				fOkToSend:=False;
			end else fMsgQ.Add(St);
		end else begin
			fMsgQ.Add(St);
		end;
	end;
end;

{===============================================================================
Plain Text Chat
===============================================================================}
Constructor tTextChat.Create(aOwner: tComponent);
begin
	inherited Create(aOwner);
	EchoOnSend:=true;
end;

procedure tTextChat.ProcessString(St: String);
begin
	if Assigned(fOnData) then fOnData(St);
end;

procedure tTextChat.Send(St: String);
begin
	if Assigned(fChatInterface) then begin
		fChatInterface.Send(St);
	end;
end;

procedure TTabEdit.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
	inherited;
	Message.Result:= Message.Result or DLGC_WANTTAB;
end;

procedure Register;
begin
	RegisterComponents('NetChat',[TTextChat,TBnetChat,TTabEdit]);
end;

end.
