{===============================================================================
ChatInterface.pas
	Battle.net Chat connection component
	written by Sarah Davis and Dan Stipp, translated from the Greetbot Code
	by Scott Coleman.
	send questions or comments to topazchat@hotmail.com

This Component manages the connection to battle.net.  Use this component
together with the Interpreters unit to manage a battle.net chat session.
Use the Connect Method to connect to battle.net (be sure to set the property
value for Port to 6112 for battle.net, and the Address property to the
Battle.net server you wish to connect to.

See the Interpreters unit for more information on how to use this component.
===============================================================================}
unit ChatInterface;

interface

uses
	Classes, WinSock, Forms, sysUtils, Dialogs, ExtCtrls, Controls, windows,
	Messages;

const
	MAXMESSAGESIZE=2048;
	BUFSIZE=16384;
	UWM_SOCKETEVENT=49151;

type
	tChatEvent=procedure of Object;
	tDataReceived=procedure(DataString: String) of Object;
	TChatInterface=Class(tWinControl)
	private
		fPort:				Word;
		fAddress:			String;
		fLoginScript:		tStringList;
		fLoggedOn:			Boolean;
		fConnected:			Boolean;
		fSocket:				Integer;
		fSuccess:			Boolean;
		fCancelled:			Boolean;
		fTimeOutCounter:	Integer;
		fOnConnect:			tChatEvent;
		fOnDisconnect:		tChatEvent;
		fOnSendError:		tChatEvent;
		fOnDataReceived:	tDataReceived;
		fOnLogin:			tChatEvent;
		fCounter:			Integer;
		fWSAData: 			tWSAData;
		fTimeOut:			integer;
		fConnectFailMsg:	String;
		fInterval:			Integer;
		fNoLineFeed:		Boolean;
		fNoReturn:			Boolean;
		fOnSocketError:	tChatEvent;
		fOnConnectClosed: tChatEvent;
		fChatString:		String;
		fBuf: 				array[0..BUFSIZE] of char;
		fBufPos: 			integer;
		procedure Receive;
		procedure ProcessData(ChatString: String);
		function  InitializeSocket: boolean;
		procedure ShutdownSocket;
		procedure SetPort(Value: Word);
		procedure SetAddress(Value: String);
		procedure SetLoginScript(Value: tStringList);
		procedure UWMSocketEvent(var Msg: tMessage); message UWM_SOCKETEVENT;
	public
		Connecting:			Boolean;
		Constructor Create(aOwner: tComponent); Override;
		Destructor  Destroy; Override;
		function 	Connect: Boolean;
		procedure	Disconnect;
		procedure	CancelConnect;
		function		Login: boolean;
		function		Send(St: String): boolean;
		property	LoggedOn: boolean
			read  fLoggedOn
			write fLoggedOn;
	published
		property NoLineFeed: boolean
			Read	fNoLineFeed
			write fNoLineFeed;
		property NoReturn: boolean
			Read	fNoReturn
			Write fNoReturn;
		property Port: Word
			Read 	fPort
			Write SetPort;
		property Address: String
			Read	fAddress
			Write SetAddress;
		property LoginScript: tStringList
			Read	fLoginScript
			Write SetLoginScript;
		property OnConnect: tChatEvent
			Read  fOnConnect
			Write fOnConnect;
		property OnDisconnect: tChatEvent
			Read	fOnDisconnect
			Write fOnDisconnect;
		property OnSendError: tChatEvent
			Read  fOnSendError
			Write fOnSendError;
		property OnDataReceived: tDataReceived
			Read	fOnDataReceived
			Write fOnDataReceived;
		property OnLogin: tChatEvent
			read	fOnLogin
			write fOnLogin;
		property OnSocketError: tChatEvent
			read  fOnSocketError
			write fOnSocketError;
		property Connected: boolean
			Read	fConnected;
		property OnConnectClosed: tChatEvent
			read  fOnConnectClosed
			write fOnConnectClosed;
	end;

procedure Register;
function RemoveChar(st: String;ch: char): String;

implementation

function RemoveChar(st: String;ch: char): String;
var
	i: integer;
begin
	Result:=St;
	i:=1;
	while i<=length(Result) do begin
		if Result[i]=Ch then begin
			Delete(Result,i,1);
			Dec(i);
		end;
		Inc(i);
	end;
end;

{===============================================================================
Chat Interface Functions
===============================================================================}
Constructor tChatInterface.Create(aOwner: tComponent);
begin
	inherited Create(aOwner);
	fPort:=0;
	fAddress:='';
	fLoginScript:=tStringList.Create;
	fLoggedOn:=False;
	fConnected:=False;
	fSocket:=INVALID_SOCKET;
	fBufPos:=0;
	fChatString:='';
	Connecting:=false;
	fSuccess:=false;
	fCancelled:=false;
	fTimeOutCounter:=0;
	fOnLogin:=Nil;
	fOnSocketError:=Nil;
	fOnConnect:=Nil;
	fOnDisconnect:=Nil;
	fOnSendError:=Nil;
	fOnDataReceived:=Nil;
	fOnConnectClosed:=nil;
	fCounter:=0;
	fTimeOut:=20;
	fInterval:=1500;
	Width:=30;
	Height:=30;
	Visible:=False;
	fConnectFailMsg:='No response from server.';
	fNoLineFeed:=true;
	fNoReturn:=True;
	if not InitializeSocket then begin
		ShowMessage('Unable to initialize WinSock');
	end;
end;

procedure tChatInterface.UWMSocketEvent(var Msg: tMessage);
begin
	if Msg.LParam=SOCKET_ERROR then begin
		if assigned(fOnSocketError) then fOnSocketError;
	end else
	Case Msg.LParamLo of
		FD_CONNECT: begin
			fConnected:=true;
			if Assigned(fOnConnect) then fOnConnect;
			Connecting:=False;
			fCounter:=0;
		end;
		FD_READ: begin
			Receive;
		end;
		FD_CLOSE: begin
			WSACancelBlockingCall;
			CloseSocket(fSocket);
			if Assigned(fOnConnectClosed) then fOnConnectClosed;
		end;
	end;
end;

function tChatInterface.Connect: boolean;
var
	Connection: tSockAddrIn;
	Host: PHostEnt;
	i: integer;
	FoundLetter: boolean;
begin
	Result:=False;
	if Connecting then Exit;
	Result:=True;
	Connecting:=True;
	with Connection do begin
		Sin_Port:=hTons(fPort);
		sin_family := AF_INET;
		FoundLetter:=false;
		for i:=1 to length(fAddress) do begin
			if ((ord(fAddress[i])<48) or (ord(fAddress[i])>57)) and
				(fAddress[i]<>'.') then begin
				FoundLetter:=true;
				Break;
			end;
		end;
		if FoundLetter then begin
			Host:=GetHostByName(pChar(fAddress));
			if Host=nil then begin
				fSuccess:=false;
				fCancelled:=True;
				exit;
			end;
			Move(Host.h_addr^^,Sin_Addr,Host.h_length);
		end else begin
			Sin_Addr.S_addr:=inet_addr(pChar(fAddress));
		end;
		fSocket:=socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (fSocket=INVALID_SOCKET) then begin
			fSuccess:=false;
			exit;
		end;
		WSAAsyncSelect(fSocket,Self.Handle,UWM_SOCKETEVENT,FD_CONNECT or FD_READ or
			FD_CLOSE);
		if ((WinSock.Connect(fSocket,Connection,sizeof(Connection)))>0) then begin
			Result:=false;
		end;
	end;
end;

procedure tChatInterface.Disconnect;
begin
	fLoggedOn:=false;
	fConnected:=false;
	fChatString:='';
	if (fSocket=INVALID_SOCKET) then exit;
	CloseSocket(fSocket);
	fSocket:=INVALID_SOCKET;
	if Assigned(fOnDisconnect) then fOnDisconnect;
end;

procedure tChatInterface.CancelConnect;
begin
	if not Connecting then exit;
	fCancelled:=True;
	WSACancelBlockingCall;
	CloseSocket(fSocket);
end;

Function tChatInterface.Send(St: String): boolean;
var
	i: integer;
	Buf: Array[0..MAXMESSAGESIZE] of Byte;
begin
	Result:=False;
	if not fConnected then Exit;
	fillchar(buf,sizeof(buf),#0);
	for i:=1 to length(st) do begin
		buf[i-1]:=ord(st[i]);
	end;
	if WinSock.Send(fSocket,buf,length(st),0)<=0 then begin
		if Assigned(fOnSendError) then fOnSendError;
	end else begin
		Result:=True;
	end;
end;

function tChatInterface.Login: boolean;
var
	i: integer;
begin
	Result:=true;
	if not fConnected then exit;
	for i:=0 to fLoginScript.Count-1 do begin
		if not Self.Send(fLoginScript[i]+#13) then begin
			Result:=False;
			break;
		end;
	end;
	if (Result=True) then begin
		fLoggedOn:=True;
		if Assigned(fOnLogin) then fOnLogin;
	end;
end;

Destructor tChatInterface.Destroy;
begin
	fOnDisconnect:=Nil;
	if fConnected then Disconnect;
	ShutdownSocket;
	fLoginScript.Free;
	inherited Destroy;
end;

function tChatInterface.InitializeSocket: boolean;
var
	ErrorCode: integer;
begin
	ErrorCode:=WSAStartup($101,fWSAdata);
	Case ErrorCode of
		WSAEINVAL: Result:=False;
		WSASYSNOTREADY: Result:=False;
		WSAVERNOTSUPPORTED: Result:=False;
		else Result:=True;
	end;
end;

procedure tChatInterface.ShutdownSocket;
begin
	WSACancelBlockingCall;
	WSACleanup;
end;

{===============================================================================
Chat Interface Property Functions
===============================================================================}
procedure tChatInterface.SetPort(Value: Word);
begin
	fPort:=Value;
end;

procedure tChatInterface.SetAddress(Value: String);
begin
	fAddress:=Value;
end;

procedure tChatInterface.SetLoginScript(Value: tStringList);
begin
	fLoginScript.Assign(Value);
end;

{===============================================================================
Chat Thread Functions
===============================================================================}
procedure tChatInterface.Receive;
var
	n,i: integer;
	St: String;
begin
  n:=Recv(fSocket,fBuf,BUFSIZE,0);
  for i:=0 to N-1 do begin
	  fChatString:=fChatString+fBuf[i];
  end;
  i:=1;
  While (i<Length(fChatString)) do begin
	  if (fChatString[i]=#10) or (fChatString[i]=#13) then begin
		  st:=Copy(fChatString,1,i);
		  Delete(fChatString,1,i);
		  i:=1;
		  ProcessData(St);
	  end else begin
		  Inc(i);
	  end;
  end;
end;

procedure tChatInterface.ProcessData(ChatString: String);
begin
	if fNoLineFeed then begin
		ChatString:=RemoveChar(ChatString,#10);
	end;
	if fNoReturn then begin
		ChatString:=RemoveChar(ChatString,#13);
	end;
	if Length(ChatString)>0 then begin
		if Assigned(fOnDataReceived) then fOndataReceived(ChatString);
	end;
end;

procedure Register;
begin
	RegisterComponents('NetChat',[TChatInterface]);
end;

end.
